'----------------------------------------------------------------------------
'
' Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
'
' Use, modification and distribution are subject to the Boost Software
' License, Version 1.0. (See http://www.boost.org/LICENSE_1_0.txt)
'
'----------------------------------------------------------------------------

Imports System
Imports System.Collections.Generic
Imports System.Text
Imports SIMCAQLib

Namespace SQPVBSample
    Class EzQ
        Public Sub New()
            Try
                simcaq = New SIMCAQ()
                m_project = Nothing
                m_model = Nothing
                components = Nothing
                numComponents = -1
                dModXCrit = -1
                hotellingsT2Crit = -1
                probLevel = -1
            Catch generatedExceptionName As System.Exception
                'e
                ' Can be that x64 application tries to use an x86 DLL.
            System.Windows.Forms.MessageBox.Show("SIMCA-Q is not registered as a COM DLL, or architecture mismatch.")

            End Try
        End Sub

        ' Open the project.
        ' @param projectPath    In: is the full path to the file.
        ' @param projectPassword   In: The password of the file. Can be NULL if not used.
        ' @return             Returns true if successful
        Public Function OpenProject(projectPath As String, projectPassword As String) As Boolean
            If m_project IsNot Nothing Then
                m_project.DisposeProject()
                m_project = Nothing
            End If
            Try
                m_project = DirectCast(simcaq.OpenProject(projectPath, projectPassword), Project)
            Catch e As System.Exception
                If TypeOf e Is System.Runtime.InteropServices.COMException Then
                    m_lastErrorMessage = simcaq.GetErrorDescription(DirectCast(e, System.Runtime.InteropServices.COMException).ErrorCode)
                End If
                Return False
            End Try

            Return True
        End Function

        ' Set the index of the model to use. By default the first model is used.
        ' @param modelIndex  In: The index of the model starting from 1 
        ' @return             Returns true if successful
        Public Function SelectModel(modelIndex As Integer) As Boolean
            Try
                System.Diagnostics.Debug.Assert(m_project IsNot Nothing)

                'project.ChangeMode(DllMode.eModePrediction);

                '///////////////////////////////////////////////////////////////////////
                ' Check that the model exists in SIMCAQ
                Dim nModels As Integer = m_project.GetNumberOfModels()

                ' Make sure that vi have some models
                If nModels < 1 Then
                    Return False
                End If

                ' Get the model number connected with the input index.
                Dim modelNumber As Integer = m_project.GetModelNumberFromIndex(modelIndex)
                m_model = DirectCast(m_project.GetModel(modelNumber), Model)

                ' Check if model is correct (=fitted) 
                Dim isFitted As Boolean = m_model.IsModelFitted() <> 0
                If Not isFitted Then
                    Return False
                End If

                ' Get results
                numComponents = m_model.GetNumberOfPredictiveComponents() + m_model.GetNumberOfXOrthogonalComponents()

                ' Set components to use
                components = DirectCast(simcaq.GetNewIntVector(1), IntVector)
                components.SetData(1, numComponents)
                probLevel = m_model.GetDefaultProbabilityLevel()
                dModXCrit = m_model.GetDModXCrit(numComponents, NormalizedState.eNormalized_True, probLevel)
                hotellingsT2Crit = m_model.GetT2RangeCrit(1, numComponents, probLevel)
            Catch e As System.Exception
                If TypeOf e Is System.Runtime.InteropServices.COMException Then
                    m_lastErrorMessage = simcaq.GetErrorDescription(DirectCast(e, System.Runtime.InteropServices.COMException).ErrorCode)
                End If
                Return False
            End Try

            Return True
        End Function

        ' Set the data of the prediction set to use. 
        ' @param quantitativeMatrix  In: A matrix with quantitative prediction data, row by row
        ' @param quantRowSize            In: Number of rows in quantitative prediction data
        ' @param quantColSize            In: Number of columns in the quantitative prediction data
        ' @param qualitativeMatrix  In: A matrix with qualitative prediction data, row by row
        ' @param qualRowSize             In: Number of rows in qualitative prediction data
        ' @param qualColSize            In: Number of columns in the qualitative prediction data
        ' @return             Returns true if successful
        Public Function Predict(quantitativeMatrix As Single(), quantRowSize As Integer, quantColSize As Integer, qualitativeMatrix As String(), qualRowSize As Integer, qualColSize As Integer) As Boolean
            Try
                ' Declare variables
                Dim prepPred As PreparePrediction = DirectCast(m_model.PreparePrediction(), PreparePrediction)
                ' Get a prepare prediction object to set data for the prediction
                Dim varVec As VariableVector = DirectCast(prepPred.GetVariablesForPrediction(), VariableVector)
                Dim numPredsetVars As Integer = varVec.GetSize()

                If numPredsetVars <> quantColSize + qualColSize Then
                    ' Must be same number of variables
                    Return False
                End If

                If quantitativeMatrix IsNot Nothing AndAlso qualitativeMatrix IsNot Nothing AndAlso quantRowSize <> qualRowSize Then
                    ' Must be same number of rows
                    Return False
                End If

                ' Set the data for the prediction
                If (quantitativeMatrix IsNot Nothing AndAlso qualitativeMatrix IsNot Nothing) Then
                    ' Must do some kind of alignment

                    For iCol As Integer = 1 To numPredsetVars
                        Dim iQualPos As Integer = 0
                        Dim iQuantPos As Integer = 0
                        Dim var As Variable = DirectCast(varVec.GetVariable(iCol), Variable)
                        Dim bIsQual As Boolean = var.IsQualitative() <> 0
                        For iRow As Integer = 1 To quantRowSize
                            If bIsQual Then
                                prepPred.SetQualitativeData(iRow, iCol, qualitativeMatrix(iRow * qualColSize + iQualPos))
                                iQualPos += 1
                            Else
                                prepPred.SetQuantitativeData(iRow, iCol, quantitativeMatrix(iRow * quantColSize + iQuantPos))

                                iQuantPos += 1
                            End If
                        Next
                    Next
                ElseIf quantitativeMatrix IsNot Nothing Then
                    Dim obsRawData As FloatMatrix = DirectCast(simcaq.GetNewFloatMatrix(quantRowSize, quantColSize), FloatMatrix)
                    obsRawData.SetMatrix(quantitativeMatrix(0))
                    prepPred.SetQuantitativeDataMatrix(obsRawData)
                ElseIf qualitativeMatrix IsNot Nothing Then
                    Dim obsRawData As StringMatrix = DirectCast(simcaq.GetNewStringMatrix(qualRowSize, qualColSize), StringMatrix)
                    obsRawData.SetMatrix(qualitativeMatrix(0))
                    prepPred.SetQualitativeDataMatrix(obsRawData)
                End If

                ' Predict
                m_prediction = DirectCast(prepPred.GetPrediction(), Prediction)
            Catch e As System.Exception
                If TypeOf e Is System.Runtime.InteropServices.COMException Then
                    m_lastErrorMessage = simcaq.GetErrorDescription(DirectCast(e, System.Runtime.InteropServices.COMException).ErrorCode)
                End If
                Return False
            End Try

            Return True
        End Function

        ' Get predicted T (scores)
        ' @return             Returns the result or null if not successful
        Public Function GetTPS() As VectorData
            System.Diagnostics.Debug.Assert(m_project IsNot Nothing)
            System.Diagnostics.Debug.Assert(m_prediction IsNot Nothing)

            Try
                Return DirectCast(m_prediction.GetTPS(components), VectorData)
            Catch e As System.Exception
                If TypeOf e Is System.Runtime.InteropServices.COMException Then
                    m_lastErrorMessage = simcaq.GetErrorDescription(DirectCast(e, System.Runtime.InteropServices.COMException).ErrorCode)
                End If
                Return Nothing
            End Try
        End Function

        ' Get T (scores)
        ' @return             Returns the result or null if not successful
        Public Function GetT() As VectorData
            System.Diagnostics.Debug.Assert(m_project IsNot Nothing)
            System.Diagnostics.Debug.Assert(m_model IsNot Nothing)

            Try
                Return DirectCast(m_model.GetT(components), VectorData)
            Catch e As System.Exception
                If TypeOf e Is System.Runtime.InteropServices.COMException Then
                    m_lastErrorMessage = simcaq.GetErrorDescription(DirectCast(e, System.Runtime.InteropServices.COMException).ErrorCode)
                End If
                Return Nothing
            End Try
        End Function

        ' Get predicted DModX
        ' @return             Returns the result or null if not successful
        Public Function GetDModXPS() As VectorData
            System.Diagnostics.Debug.Assert(m_project IsNot Nothing)
            System.Diagnostics.Debug.Assert(m_prediction IsNot Nothing)
            Try
            Return DirectCast(m_prediction.GetDModXPS(components, NormalizedState.eNormalized_True, ModelingPowerWeightedState.eModelingPowerWeighted_False), VectorData)
            Catch e As System.Exception
                If TypeOf e Is System.Runtime.InteropServices.COMException Then
                    m_lastErrorMessage = simcaq.GetErrorDescription(DirectCast(e, System.Runtime.InteropServices.COMException).ErrorCode)
                End If
                Return Nothing
            End Try
        End Function

        ' Get predicted Y
        ' @return             Returns the result or null if not successful
        Public Function GetYPredPS() As VectorData
            System.Diagnostics.Debug.Assert(m_project IsNot Nothing)
            System.Diagnostics.Debug.Assert(m_prediction IsNot Nothing)
            Try
                Return DirectCast(m_prediction.GetYPredPS(numComponents, UnscaledState.eUnscaled_False, BacktransformedState.eBacktransformed_True, Nothing), VectorData)
            Catch e As System.Exception
                If TypeOf e Is System.Runtime.InteropServices.COMException Then
                    m_lastErrorMessage = simcaq.GetErrorDescription(DirectCast(e, System.Runtime.InteropServices.COMException).ErrorCode)
                End If
                Return Nothing
            End Try
        End Function

        ' Get predicted T2
        ' @return             Returns the result or null if not successful
        Public Function GetT2RangePS() As VectorData
            System.Diagnostics.Debug.Assert(m_project IsNot Nothing)
            System.Diagnostics.Debug.Assert(m_prediction IsNot Nothing)
            Try
                Return DirectCast(m_prediction.GetT2RangePS(1, numComponents), VectorData)
            Catch e As System.Exception
                If TypeOf e Is System.Runtime.InteropServices.COMException Then
                    m_lastErrorMessage = simcaq.GetErrorDescription(DirectCast(e, System.Runtime.InteropServices.COMException).ErrorCode)
                End If
                Return Nothing
            End Try
        End Function

        ' Get the model T2Range critical limit
        ' @return             Returns the result or null if not successful
        Public Function GetT2RangeCrit() As Single
            System.Diagnostics.Debug.Assert(m_project IsNot Nothing)
            Try
                Return m_model.GetT2RangeCrit(1, numComponents, -1)
            Catch e As System.Exception
                If TypeOf e Is System.Runtime.InteropServices.COMException Then
                    m_lastErrorMessage = simcaq.GetErrorDescription(DirectCast(e, System.Runtime.InteropServices.COMException).ErrorCode)
                End If
                Return 0
            End Try
        End Function

        ' Close the project. This is also done automatically by the 
        ' destructor when the object goes out of scope.
        ' @return             Returns non zero if no error.
      Public Function CloseProject() As Boolean
         If m_project IsNot Nothing Then
            m_project.DisposeProject()
            m_project = Nothing
         End If
         Return True
      End Function

        Protected simcaq As SIMCAQ
        Protected m_project As Project

        Public ReadOnly Property Project() As Project
            Get
                Return m_project
            End Get
        End Property
        Protected m_prediction As Prediction
        Public ReadOnly Property Prediction() As Prediction
            Get
                Return m_prediction
            End Get
        End Property
        Protected m_model As Model
        Public ReadOnly Property Model() As Model
            Get
                Return m_model
            End Get
        End Property
        Public ReadOnly Property LastErrorMessage() As [String]
            Get
                Return m_lastErrorMessage
            End Get
        End Property

        Protected components As IntVector
        Protected numComponents As Integer
        Protected dModXCrit As Single
        Protected hotellingsT2Crit As Single
        Protected probLevel As Single
        Protected m_lastErrorMessage As [String]
    End Class
End Namespace
